/*
 * ConstraintProblem.java
 *
 * 
 */
 package DisCSP.CSP;

/**
 * @ version 1.0
 *
 * @ author Nocerino Francesca
 *
 * @ since JDK 1.4
 *
 */


import DisCSP.CSP.Constraint.*;
import DisCSP.Exception.*;

public class ConstraintProblem
{
	private VariableVector vars;
	private BinaryConstraintVector constrVect;

 	/** Costruttore di ConstraintProblem
     * 
     * 
     */	
	public ConstraintProblem()
	{
		vars= new VariableVector();
		constrVect=new BinaryConstraintVector();
	}
	
	/** Restituisce l'i-esima variabile del problema
	 *
	 * @param i l'indice della variabile
     * @return l'i-esima variabile del problema
     * 
     */		
	public  Variable variableAt(int i)
	{
		return vars.variableAt(i);
	}
	
	/** Resituisce una variabile dato il suo nome, che  un identificatore univoco per le variabili
	 *
	 * @param name il nome della variabile
     * @return la variabile
     * 
     */		
	public  Variable getVarByName(String name) throws UnknownVariableException
	{
		for(int i=0;i<numberOfVariable();i++)
		{
			if(vars.variableAt(i).getName().equals(name)) return variableAt(i);
		}
		
			throw new UnknownVariableException(name);  
	}
	
	/** Permette di aggiungere una variabile al problema
	 *
	 * @param v la variabile
     * 
     */			
	public  void addVariable(Variable v) throws ExistentVarException
	{
		
		for(int i=0;i<numberOfVariable();i++)
		{
			if(vars.variableAt(i).getName().equals(v.getName())) throw new ExistentVarException(v.getName());
		}

		vars.add(v);
	}
	
	/** Permette di rimuovere una variabile dal problema
	 *
	 * @param v la variabile
     * 
     */			
	public  void removeVariable(Variable v)
	{
		vars.remove(v);
		for(int i=0;i<numberOfConstraint();i++)
		{
			Variable[] vapp= constraintAt(i).getVariables();
			if(vapp[0].getName().equals(v.getName())||vapp[1].getName().equals(v.getName()))
			removeConstraint(constraintAt(i));

		}
	}
	
	/** Restituisce il numero di variabili del problema
	 *
     * @return il numero di variabili del CP
     * 
     */		
	public  int numberOfVariable()
	{
		return vars.size();
	}
	
	/** Restituisce l'i-esimo vincolo del problema
	 *
	 * @param i l'indice del vincolo
     * @return l'i-esimo vincolo
     * 
     */		
	public  BinaryConstraint constraintAt(int i)
	{
		return (BinaryConstraint) constrVect.elementAt(i);
	}
	
	/**  Permette di aggiungere un vincolo al problema
	 *
	 * @param constr il vincolo da aggiungere
     * 
     */			
	public  void addConstraint(BinaryConstraint constr) throws UnknownVariableException
	{
		Variable[] v= constr.getVariables();
		if(!vars.isPresent(v[0]))
			throw new UnknownVariableException(v[0].getName());
		if(!vars.isPresent(v[1]))
			throw new UnknownVariableException(v[1].getName());  
		constrVect.add(constr);
	}
	
	/** Permette di rimuovere un vincolo dal problema
	 *
	 * @param constr il vincolo da eliminare
     * 
     */			
	public  void removeConstraint(BinaryConstraint constr)
	{
		constrVect.remove(constr);
	}
	
	/** Restituisce il numero di vincoli del problema
	 *
     * @return il numero di vincoli del CP
     * 
     */		
	public  int numberOfConstraint()
	{
		return constrVect.size();
	}
	
	/** Stampa il problema
	 *
     * @return una stringa che rappresenta il problema
     * 
     */		
	public  String print()
	{
		String prob="Variables\n";
		for(int i=0;i<numberOfVariable();i++)
		{
			prob+=variableAt(i).print()+"\n";
		}
		prob+="\nConstraints\n";
		for(int i=0;i<numberOfConstraint();i++)
		{
			prob+=constraintAt(i).print()+"\n";
		}
		return prob;
	}
	
	/** Permette di verificare l'esistenza di una certa variabile
	 *
	 * @param la variabile
     * @return true se la variabile  presente, false altrimenti
     * 
     */		
	public  boolean isVarPresent(Variable v)
	{
		return vars.isPresent(v);
	}
	
	/** Permette di partizionare un CP
	 *
	 * @param nPart il numero di classi in cui si vuole partizionare il CP
     * @return igli insiemi in cui sono state partizionate le variabili
     * 
     */		
	public String[] partition(int nPart)
	{
		VariableVector[] parts= new VariableVector[nPart];
		double[] pref=new double[nPart];
		int nVar=numberOfVariable(); 
		double nVarD=nVar;
		for(int i=0;i<nPart;i++)
		{
			parts[i]= new VariableVector();
		}
		
		for(int j=0;j<nVar;j++)
		{
			Variable currentVar=variableAt(j);
			 
			for(int i=0;i<nPart;i++)
			{
				pref[i]+=0.35*((nVarD-parts[i].size())/nVar);
				pref[i]+=0.65*relation(parts,currentVar,i);
				System.out.println("Var "+ currentVar.getName()+" pref per agent"+i+" "+pref[i]);
			}
			
			parts[maxIndex(pref)].addVariable(currentVar);
		}
		
		String[] ris= new String[nPart];
		
		for(int i=0;i<nPart;i++)
		{
			String set="";
			for(int j=0;j<parts[i].size();j++)
			{
				set+=parts[i].variableAt(j).getName();
				if(j!=parts[i].size()) set+=" ";
			}
			ris[i]=set;
		}

		return ris;
		
	}
	
	private int maxIndex(double val[])
	{
		int maxInd=0;
		double max=val[0];
		for(int i=1;i<val.length;i++)
		{
			if(val[i]>max)
			{
				max=val[i];
				maxInd=i;
			}
		}
		return maxInd;
	}
	
	private double relation(VariableVector[] sets, Variable ins, int currSet)
	{
		double rel=0,  rel1=0, rel2=0;
		double yes=0, no=0;	 
		for(int k=0;k<numberOfConstraint();k++)
		{
			Variable[] varConstr=constraintAt(k).getVariables();
				
			if( varConstr[0]==ins|| varConstr[1]==ins)
			{	
				for(int i=0;i<sets.length;i++)
				{	
					if (sets[i].contains(varConstr[0])||sets[i].contains(varConstr[1]) )
					{
						if (i==currSet)
						{
							rel1+=value(constraintAt(k));
							yes++;
						}
						else
						{
							rel2+=value(constraintAt(k));
							no++;
						}

					}
			
				}	
			}
		}	
		if(rel1!=0) rel1= rel1/yes;
		if(rel2!=0) rel2= rel2/no;
		rel= ( rel1-rel2 +1)/2;
		return rel;
	}
	
	private double value(BinaryConstraint constr)
	{
		if(constr.getClass().getName().equals("DisCSP.CSP.Constraint.EqualConstraint") )
			return 3;
			
		if(constr.getClass().getName().equals("DisCSP.CSP.Constraint.NotEqualConstraint") )
			return 1;
			
		return 2;	
	}

}